/*
 * NAME
 *     vc:  version control
 *
 * SYNOPSIS
 *     vc [-c char] [-s] [-x] [-w] [variable=value]... [file]...
 *
 * DESCRIPTION
 *     See man page (vc.1)
 *
 * MODIFICATION HISTORY
 * Mnemonic     Rel Date    Who
 * vc           1.0 020924  mpw
 *      Created.
 *
 * Copyright (C) 2002, 2003, 2004, 2023 Mark Willson.
 *
 * This file is part of the vc program.
 *
 * vc is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * vc is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with vc; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <sys/stat.h>
#include "vc.h"
#include "op.h"
#include "lexer.h"
#include "var.h"
#include "parse.h"
#include "symtab.h"

/* GLOBALS */
char cchar = '%';       /* vc statement control character */
bool silent = false;    /* true if warning messages to be suppressed */
bool debug = false;     /* true if diagnostics required */
bool warn = false;      /* true if expansion warnings required */

/* define include file stack */
struct {
    char *name;
    int n;
} incstk[INCSTKSZ];
int incsp = 0;

/* push filename and line count on stack */
bool
incpush(char *s, int n)
{
    if (incsp < INCSTKSZ) {
        incstk[incsp].name = s;
        incstk[incsp++].n = n;
        return true;
    }
    else return false;
}

/* pull filename and line count from stack */
int
incpull(char **s, int *n)
{
    if (incsp > 0) {
        *s = incstk[--incsp].name;
        *n = incstk[incsp].n;
        return true;
    }
    else return false;
}

void
vcerr(char *s, char *p)
{
    char buf[BUFSZ+1];

    if (p == NULL)
        fprintf(stderr,"vc: %s\n", s);
    else {
        sprintf(buf,"vc: %s\n", s);
        fprintf(stderr, buf, p);
    }
    exit(EXIT_FAILURE);
}

void
errmsg(int vcerrno, char *s, int nrec, char *fn)
{
    const char *msg;

    msg = parse_geterrmsg(vcerrno);
    fprintf(stderr,"vc: %s at line %d (%s)\n", msg, nrec, fn);
    vcerr("offending line: \"%s\"\n", s);
}

void
warnmsg(int warnings, char *s, int nrec, char *fn)
{
    const char *msg;

    if ((warnings & (W_LONECC | W_UNDEF)) == (W_LONECC | W_UNDEF)) {
        msg = "unpaired control char AND undefined variable encountered";
    }
    else if (warnings & W_LONECC) {
        msg = "unpaired control char encountered";
    }
    else if (warnings & W_UNDEF) {
        msg = "undefined variable encountered";
    }
    fprintf(stderr, "vc: %s at line %d (%s)\n", msg, nrec, fn);
}

/* Expand vars referenced in src string. Return expanded string.  If
 * warnings is returned non-zero, one or more expansion warnings
 * occurred.
 */
char *
expvar(char *src, int* warnings) {
    static char expstr[BUFSZ*2];
    char *s = src,
        *t,
        varname[BUFSZ+1];
    VAR *value;

    t = expstr;
    *warnings = 0;

    if (*s == '\\' && *(s+1) == cchar) s++;

    while (*s) {
        if (*s == cchar) {
            char *end = strchr(s+1, cchar);
            if (end == NULL) {
                *warnings |= W_LONECC;
            }
            else {
                strncpy(varname, s+1, end-s-1);
                if (end-s > BUFSZ)
                    varname[BUFSZ] = '\0';
                else
                    varname[end-s-1] = '\0';
                /* expression? */
                if (varname[0] == '(' && varname[end-s-2] == ')') {
                    varname[end-s-2] = '\0';
                    lex_init(varname+1);
                    (void) lexer();
                    value = parse_expression(0);
                }
                else if (!(value = sym_get(varname))) {
                    *warnings |= W_UNDEF;
                    strncpy(t, s, end-s+1);
                    *(t + (end-s+1)) ='\0';
                }
                if (value)  {
                    sprintf(t, "%s", var_sprint(value));
                }
                s = s + (end-s+1);
                t = t + strlen(t);
                if (t > expstr + (BUFSZ*2)) return NULL;
                continue;
            }
        }
        *t++ = *s++;
        if (t > expstr + (BUFSZ*2)) return NULL;
    }
    *t = '\0';
    return expstr;
}

FILE *
open_file(char *fn)
{
    struct stat sp;
    FILE *fp = NULL;

    if (strcmp(fn, "stdin") == 0) {
        fp = stdin;
    }
    else {
        stat(fn, &sp);
        if (S_ISREG(sp.st_mode))
            fp = fopen(fn, "r");
    }
    return fp;
}

void
process(char* fn)
{
    FILE *fp;
    int nrec = 0,
        i, action,
        warnings = 0;
    bool expand = true,
         skip = false;
    char buf[BUFSZ+1],
         arg[BUFSZ+1], *fcp,
         *cp, *expanded;

    DEBUGV("process: file %s\n",fn);
    fp = open_file(fn);
    if (fp == NULL) vcerr("unable to open file %s",fn);

    do {
        if (fgets(buf,BUFSZ,fp) == NULL) {
            if (fp != stdin) fclose(fp);
            fcp = fn;
            if (!incpull(&fn,&nrec)) break;
            free(fcp);
            fp = open_file(fn);
            if (fp == NULL) vcerr("unable to re-open file %s",fn);
            if (fp == stdin)
                fgets(buf, BUFSZ, fp);
            else
                for (i=0;i<=nrec;i++)
                    if (!fgets(buf,BUFSZ,fp)) {
                        buf[0] = '\0';
                    }
        }
        if (buf[strlen(buf)-1] == '\n')
            buf[strlen(buf)-1] = '\0'; /* clobber newline */
        cp = strrchr(buf, '\r');
        if (cp) {
            *cp = '\0'; /* and CR if present */
        }
        nrec++;

        if (buf[0] == cchar) {
            action = parse_statement(buf+1, arg, BUFSZ);
            DEBUGV("process: f/l %s/%d action = %d\n", fn, nrec, action);
            switch (action) {
            case A_SKIP:
                skip = true;
                break;
            case A_COPY:
                skip = false;
                break;
            case A_EXPON:
                expand = true;
                break;
            case A_EXPOFF:
                expand = false;
                break;
            case A_INC:
                if (fp != stdin) fclose(fp);
                if (!incpush(fn,nrec))
                    vcerr("include files nested too deep at %s", arg);
                fp = fopen(arg, "r");
                if (fp == NULL)
                    vcerr("unable to open include file %s\n", arg);
                fn = strdup(arg);
                nrec = 0;
                break;
            case A_MSG:
                if (!silent) {
                    expanded = expvar(arg, &warnings);
                    if (!expanded)
                        errmsg(E_BADEXP, arg, nrec, fn);
                    else
                        fprintf(stderr,"%s\n", expanded);
                }
                break;
            case A_ERR:
                expanded = expvar(arg, &warnings);
                if (!expanded)
                    errmsg(E_BADEXP, arg, nrec, fn);
                else
                    fprintf(stderr,"%s\nERROR:  on line %d (%s)\n",
                            expanded, nrec, fn);
                exit(EXIT_FAILURE);
            case A_NOP:
                break;
            case A_DUMPSYM:
                sym_dmp();
                break;
            default:
                errmsg(action,buf,nrec,fn);
                break;
            }
        }
        else {
            if (!skip) {
                if (expand) {
                    char *expanded;
                    int warnings;
                    expanded = expvar(buf, &warnings);
                    if (!expanded) vcerr("unable to expand line: %s", buf);
                    printf("%s\n", expanded);
                    if (warnings && warn) {
                        warnmsg(warnings, buf, nrec, fn);
                    }
                }
                else {
                    printf("%s\n", buf);
                }
            }
        }
    } while (true);

    /* free all defined vars */
    var_free();
    return;
}

int
main(int argc,char *argv[])
{
    VAR value;
    char *vp,* tp, *s;

    /* handle command optionsqq */
    while (--argc > 0 && (*++argv)[0] == '-') {
        for (s=argv[0]+1;*s != '\0'; s++) {
            switch (*s) {
            case 's':
                silent = true;
                break;
            case 'w':
                warn = true;
                break;
            case 'x':
                debug = true;
            }
        }
    }

    /* var command line definitions */
    sym_init();
    while (argc > 0 && (vp = strchr(*argv, '='))) {
        *vp++ = '\0';
        value.type = S_INT;
        value.val.ival = strtol(vp,&tp,10);
        if (vp == tp) {
            value.type = S_STR;
            value.val.pval = strdup(vp);
            if (!value.val.pval)
                vcerr("no memory for value of var: %s", *argv);
        }
        if (!sym_set(*argv, &value))
            vcerr("unable to create variable: %s", *argv);
        argc--;
        argv++;
    }
    /* process files */
    if (argc == 0) {
        process("stdin");
    }
    else {
        while (argc-- > 0) {
            process(*argv++);
        }
    }
    return EXIT_SUCCESS;

}
