//

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include "lexer.h"
#include "var.h"
#include "parse.h"

static char *expr = NULL;
static int expr_size = 132;

struct st_mem {
    VAR * var;
    struct st_mem * next;
};
typedef struct st_mem MEM;

static MEM *allocated = NULL;

static MEM *
allocate(VAR *v)
{
    MEM *new = malloc(sizeof(MEM));

    if (!new) return NULL;

    if (!allocated) {
        allocated = new;
        allocated->var = v;
        allocated->next = NULL;
    }
    else {
        new->var = v;
        new->next = allocated;
        allocated = new;
    }
    return new;
}

void
var_free(void)
{
    MEM *link = allocated,
        *tmp;
    VAR *v;
    int free_count = 0;

    /* Free vars */
    while (link != NULL) {
        v = link->var;
        if (v->type == S_STR) {
            free(v->val.pval);
        }
        free(v);
        tmp = link->next;
        free(link);
        link = tmp;
        free_count++;
    }
    allocated = NULL;
    if (debug)
        fprintf(stderr, "Freed vars: %d\n", free_count);
}

char *
var_sprint(VAR *value)
{
    if (!expr) expr = malloc(expr_size);

    if (expr == NULL) {
        expr = "no memory";
    }
    else {
        char *format;
        switch (value->type) {
        case S_INT:
            format = "%d";
            break;
        case S_STR:
            format = "%s";
            break;
        case S_SYM:
            format = "(var %s)";
            break;
        default:
            format = "eh?";
        }
        snprintf(expr, expr_size-1, format, value->val);
    }
    return expr;
}

VAR *
var_new(void)
{
    VAR *v = malloc(sizeof(VAR));
    MEM *m = allocate(v);

    if (!(v && m)) {
        fprintf(stderr, "var_new: unable to allocate memory");
        exit(EXIT_FAILURE);
    }
    v->type = S_UNDEF;
    v->val.ival = 0;
    return v;
}

/*------------------------------------------------------------------------
 * perform operations on variables
 */

VAR *
var_add(VAR *v1, VAR *v2)
{
    int tlen;
    char *ns;
    VAR *v = var_new();

    if (v1->type != v2->type) {
        parse_error = E_TYPEDIFF; /* type mismatch */
    }
    else {
        switch (v1->type) {
            case S_INT:
                v->type = S_INT;
                v->val.ival = v1->val.ival + v2->val.ival;
                break;
            case S_STR:
                tlen = strlen(v1->val.pval)+strlen(v2->val.pval);
                ns = (char *) malloc(tlen+1);
                strcpy(ns,v1->val.pval);
                strncat(ns,v2->val.pval, tlen);
                v->type = S_STR;
                v->val.pval = ns;
                break;
            default:
               parse_error = E_BADTYPE;
        }
    }
    return v;
}

VAR *
var_sub(VAR *v1, VAR *v2)
{
    VAR *v = var_new();

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF; /* type mismatch */
    }
    else {
        switch (v1->type) {
            case S_INT:
                v->type = S_INT;
                v->val.ival = v1->val.ival - v2->val.ival;
                break;
            default:
               parse_error = E_BADTYPE;
        }
    }
    return v;
}

VAR *
var_mul(VAR *v1, VAR *v2)
{
    VAR *v = var_new();

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF; /* type mismatch */
    }
    else {
        switch (v1->type) {
            case S_INT:
                v->type = S_INT;
                v->val.ival = v1->val.ival * v2->val.ival;
                break;
            default:
               parse_error = E_BADTYPE;
        }
    }
    return v;
}

VAR *
var_div(VAR *v1, VAR *v2)
{
    VAR *v = var_new();

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF; /* type mismatch */
    }
    else {
        switch (v1->type) {
            case S_INT:
                v->type = S_INT;
                if (v2->val.ival == 0) {
                   parse_error = E_DIVZERO;
                }
                else {
                    v->val.ival = v1->val.ival / v2->val.ival;
                }
                break;
            default:
               parse_error = E_BADTYPE;
        }
    }
    return v;
}

VAR *
var_cmp(int op, VAR *v1, VAR *v2)
{
    VAR *v = var_new();
    int result = 0, cmpval;

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF;
       return v;
    }
    if (v1->type == S_STR) {
        cmpval = strcmp(v1->val.pval,v2->val.pval);
        switch (op) {
            case '<':
                result = cmpval < 0;
                break;
            case '>':
                result = cmpval > 0;
                break;
            case S_EQ:
                result = cmpval == 0;
                break;
            case S_NEQ:
                result = cmpval != 0;
                break;
            case S_LTEQ:
                result = cmpval <= 0;
                break;
            case S_GTEQ:
                result = cmpval >= 0;
                break;
            default:
               parse_error = E_BADEXP;
        }
    }
    else if (v1->type == S_INT) {
        switch (op) {
            case '<':
                result = v1->val.ival < v2->val.ival;
                break;
            case '>':
                result = v1->val.ival > v2->val.ival;
                break;
            case S_EQ:
                result = v1->val.ival == v2->val.ival;
                break;
            case S_NEQ:
                result = v1->val.ival != v2->val.ival;
                break;
            case S_LTEQ:
                result = v1->val.ival <= v2->val.ival;
                break;
            case S_GTEQ:
                result = v1->val.ival >= v2->val.ival;
                break;
            default:
               parse_error = E_BADEXP;
        }
    }
    else {
       parse_error = E_BADTYPE;
    }
    v->type = S_INT;
    v->val.ival = result;
    return v;
}

VAR *
var_and(VAR *v1, VAR *v2)
{
    VAR* v = var_new();

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF;
       return v;
    }
    if (v1->type == S_INT) {
        v->type = S_INT;
        v->val.ival = v1->val.ival && v2->val.ival;
    }
    else {
       parse_error = E_BADEXP;
    }
    return v;
}

VAR *
var_or(VAR *v1, VAR *v2)
{
    VAR* v = var_new();

    if (v1->type != v2->type) {
       parse_error = E_TYPEDIFF;
       return v;
    }
    if (v1->type == S_INT) {
        v->type = S_INT;
        v->val.ival = v1->val.ival || v2->val.ival;
    }
    else {
       parse_error = E_BADEXP;
    }
    return v;
}

VAR *
var_nud_op(char op, VAR* v1)
{
    VAR *v = var_new();

    if (v1->type != S_INT) {
        parse_error = E_BADINT;
    }
    else {
        switch (op) {
        case '!':
            v->val.ival = ! v1->val.ival;
            break;
        case '-':
            v->val.ival = - v1->val.ival;
            break;
        case '+':
            v->val.ival = v1->val.ival;
            break;
        default:
            parse_error = E_BADEXP;
        }
        v->type = S_INT;
    }
    return v;
}
