/* symtab.c: handles storing and retrieving of variable names

   Contains the following public routines:

   sym_def - defines a variable name (puts in symbol table)
   sym_set - sets a value for a defined variable name
   sym_get - returns a value for a defined variable name
   sym_dmp -  dumps contents of symbol table

   Variable types are int or character.

   created 27/09/90 MPW

*/

#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include "lexer.h"
#include "var.h"
#include "symtab.h"

enum {
    HASHSIZE = 101
};

extern bool debug;

struct s_sym {
    char *name;
    struct s_var *value;
    struct s_sym *next;
};
typedef struct s_sym SYM;

/* define symbol table */
static SYM *symtab[HASHSIZE];

static int
hash(char *s)       /* return hash value for string */
{
    int hashval;

    for (hashval=0; *s != '\0'; ) hashval += *s++;
    return hashval%HASHSIZE;
}

SYM *
lookup(char *s) /*return pointer to symbol table entry for s */
{
    SYM *sp;

    for (sp = symtab[hash(s)];sp != NULL; sp = sp->next)
        if (strcmp(s,sp->name) == 0) return sp;
    return NULL;
}

void
sym_init(void)
{
    int i;
    static bool init_done = false;

    if (!init_done) {
        for (i=0;i<HASHSIZE;i++) symtab[i] = NULL;
        init_done = true;
    }
}

SYM *
sym_def(char *s)         /* define variable name s */
{
    SYM *sp;
    int hashval;

    if ((sp=lookup(s)) == NULL) {
        sp = (SYM *) malloc(sizeof (SYM));
        if (sp == NULL) return NULL;
        sp->value = (struct s_var *) malloc(sizeof (VAR));
        if (sp->value == NULL) return NULL;
        if ((sp->name=strdup(s)) == NULL) return NULL;
        hashval = hash(sp->name);
        sp->next = symtab[hashval];
        symtab[hashval] = sp;
        sp->value->type = S_INT;
        sp->value->val.ival = 0;
    }
    return sp;
}

VAR *
sym_set(char *s, VAR *value)  /* set variable name s * to value val */
{
    SYM *sp;

    if ((sp=lookup(s)) == NULL && ((sp = sym_def(s)) == NULL)) return NULL;
    if (sp->value->type == S_STR) free(sp->value->val.pval);
    sp->value->type = value->type;
    if (value->type == S_STR) {
        if ((sp->value->val.pval=strdup(value->val.pval)) == NULL)
            return NULL;
    }
    else if (value->type == S_INT)
        sp->value->val.ival = value->val.ival;
    else
        return NULL;
    return sp->value;
}

VAR *
sym_get(char *s)     /* get value of variable s */
{
    SYM *sp;

    if ((sp=lookup(s)) == NULL) {
        return NULL;
    }
    else {
        return sp->value;
    }
}


void
sym_dmp()         /* dump symbol table entries */
{
    int i;
    SYM *sp;

    fprintf(stderr,"%-24s%-8s%s\n", "Name", "Type", "Value");
    for (i=0;i<HASHSIZE;i++) {
        sp = symtab[i];
        while (sp != NULL) {
            fprintf(stderr,"%-24s",sp->name);
            if (sp->value->type == S_INT)
                fprintf(stderr,"%-8s%d\n", "int", sp->value->val.ival);
            else if (sp->value->type == S_STR)
                fprintf(stderr,"%-8s%s\n", "char", sp->value->val.pval);
            else
                fprintf(stderr,"%-8s%d\n", "eh?", sp->value->type);
            sp = sp->next;
        }
    }
}
