/*
  Pratt-based parser implementation.

  In the topdown algorithm, each token has two associated functions,
  Called "nud" and "led", and an integer value called "lbp".

  The "nud" function (for null denotation) is used when a token
  appears at the beginning of a language construct, and the "led"
  function (left denotation) when it appears inside the construct (to
  the left of the rest of the construct, that is).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <errno.h>
#include "lexer.h"
#include "var.h"
#include "symtab.h"
#include "parse.h"
#include "op.h"

extern char cchar;      /* control character */
extern bool debug;      /* set true if debug info required */

static int exec = 1;    /* set non-zero if control statements
                         * should be executed */
static int elifcnt;     /* count of elif levels */

int parse_error = 0;    /* contains error code, if non-zero */

/* Keyword constants */
enum {
    K_SET,
    K_IF,
    K_ELSE,
    K_ELIF,
    K_END,
    K_INC,
    K_ON,
    K_OFF,
    K_CTL,
    K_MSG,
    K_ERR,
    K_DUMPSYM};

/* define keyword table */
static struct keyentry {
    char *name;
    int keyno;
    int action;
} keytab[] = {
    { "set",K_SET,A_NOP },
    { "if",K_IF,0 },
    { "else",K_ELSE,0 },
    { "elif",K_ELIF,0 },
    { "end",K_END,0 },
    { "inc",K_INC,A_INC },
    { "on",K_ON,A_EXPON },
    { "off",K_OFF,A_EXPOFF },
    { "ctl",K_CTL,A_NOP },
    { "msg",K_MSG,A_MSG },
    { "err",K_ERR,A_ERR },
    { "dumpsym",K_DUMPSYM,A_DUMPSYM }
};

#define NKEY (sizeof(keytab)/sizeof(struct keyentry))

static char *errtab[] = {
	"no such message",
	"unable to define variable",
	"unterminated string",
	"illegal symbol",
	"variable expected",
	"operand types do not match",
	"illegal type in expression",
	"unable to assign value to variable",
	"string expected",
	"illegal expression",
	"attempt to divide by zero",
	"insufficient memory",
	"integer expected",
    "unknown keyword",
    "stack overflow",
    "stack underflow",
    "stack empty",
    "use of unset variable",
    "internal error: undefined value"
};

#define MAX_ENUM (sizeof(errtab)/sizeof(char *))

/*------------------------------------------------------------------------
 * stack related stuff
 */

#define STKSZ 10

static int stk[STKSZ];
static int sp = 0;

static int
push(int i)
{
    if (sp < STKSZ)
        stk[sp++] = i;
    else
        parse_error = E_STKOVR;
    return(0);
}

static int
pull()
{
    if (sp > 0 )
        return(stk[--sp]);
    else
        parse_error = E_STKUDR;
    return(0);
}

static int
top()
{
    if (sp > 0)
        return(stk[sp-1]);
    else
        parse_error = E_STKEMP;
    return(0);
}


/*------------------------------------------------------------------------
 * Parser routines
 */

VAR *
make_var(struct st_lex token)
{
    VAR *new = var_new();
    if (!new) return NULL;

    new->type = token.sym;
    switch (token.sym) {
    case S_INT:
        new->val.ival = strtol(token.tok, NULL, 10);
        if (errno != 0) parse_error = E_BADINT;
        break;
    case S_STR:
    case S_SYM:
        new->val.pval = strdup(token.tok);
        break;
    }
    return new;
}

static VAR *
nud(struct st_lex token)
{
    DEBUGV("nud: token: %s\n", token.tok);
    if (op_find(token.sym)) {
        VAR *right = parse_expression(100);
        return var_nud_op(token.sym, right);
    }
    else if (token.sym == '(') {
        VAR *exp = parse_expression(0);
        if (!lex_match(')')) parse_error = E_BADEXP;
        DEBUGV("nud: exp: %s exp->type: %d\n", var_sprint(exp), exp->type);
        if (exp->type == S_SYM) exp = sym_get(exp->val.pval);
        return exp;
    }
    else if (token.sym == S_USTR) {
        parse_error = E_BADSTR;
    }
    return make_var(token);
}

static VAR *
led(struct st_lex token, VAR *left)
{
    struct st_op *op;
    VAR *right;
    int rbp;

    DEBUGV("led: token: %s, left: %s\n", token.tok, var_sprint(left));
    if (!(op = op_find(token.sym))) {
        return make_var(token);
    }
    /* rbp reduced for right associating operators */
    rbp =  (op->op == '=') ? op->lbp-1 : op->lbp;

    right =  parse_expression(rbp);
    if (!right) {
        if (!parse_error) parse_error = E_BADEXP;
        return NULL;
    }

    /* symbol assignment? */
    if (op->op == '=') {
        if (left->type == S_SYM) {
            sym_set(left->val.pval, right);
            return right;
        }
        parse_error = E_BADEXP;
        return NULL;
    }
    /* not symbol assignment; replace symbol by value */
    if (left->type == S_SYM) {
        left = sym_get(left->val.pval);
        if (!left) {
            parse_error = E_BADSYM;
            return NULL;
        }
    }
    DEBUGV("led: left: %s, token: %s, right: %s\n", var_sprint(left),
           token.tok, var_sprint(right));
    switch (op->op) {
    case '=':
    case '+':
        return var_add(left, right);
    case '-':
        return var_sub(left, right);
    case '*':
        return var_mul(left, right);
    case '/':
        return var_div(left, right);
    case S_EQ:
    case S_NEQ:
    case S_GTEQ:
    case S_LTEQ:
    case '<':
    case '>':
        return var_cmp(op->op, left, right);
    case '&':
        return var_and(left, right);
    case '|':
        return var_or(left, right);
    default:
        parse_error = E_BADSYM;
        return NULL;
    }
    return make_var(token);
}

static int
lbp(struct st_lex token)
{
    struct st_op *op;

    if ((op = op_find(token.sym))) {
        return op->lbp;
    }
    return 0;
}

VAR *
parse_expression(int rbp)
{
    struct st_lex t;
    VAR *left;

    DEBUGV("pe: exp: rbp: %d\n", rbp);
    t = lex;
    (void) lexer();
    DEBUGV("pe: t: %s, next_token: %s\n", t.tok, lex.tok);
    left = nud(t);
    DEBUGV("pe: left: %s; lex: %s; lbp(lex): %d\n",
           var_sprint(left), lex.tok, lbp(lex));

    while (rbp < lbp(lex)) {
        t = lex;
        (void) lexer();
        left = led(t, left);
    }
    if (parse_error) return NULL;
    if (left->type == S_SYM) {
        left = sym_get(left->val.pval);
        if (!left) parse_error = E_BADSYM;
    }
    else if (left->type == S_UNDEF) {
        parse_error = E_INTERNAL;
    }
    return left;
}

/*
 * Parse vc statement line in buf (cchar omitted). Returns statement
 * action number, or error number if an error occured. For statements
 * that have an argument, it is copied into arg, limited to argsize.
 */

int
parse_statement(char *buf, char *arg, int argsize)
{
    int rval,i;
    VAR *value;
    char varname[LEXTOKSIZ+1];

    *arg = '\0';
    lex_init(buf);
    /* read keyword */
    (void) lexer();

    DEBUGV("keyword: %s, type %d\n", lex.tok, lex.sym);
    for (i=0;i<NKEY;i++)
        if (strcmp(keytab[i].name, lex.tok) == 0)
            break;
    if (i >= NKEY)
        return E_BADKEY;
    rval = keytab[i].action;
    (void) lexer();

    switch (keytab[i].keyno) {
    case K_SET:
        if (exec) {
            if (lex.sym != S_SYM) return E_SYMEXP;
            strncpy(varname, lex.tok, LEXTOKSIZ);
            (void) lexer();
            if (!lex_match('=')) return E_BADEXP;
            if  (!(value = parse_expression(0)))
                return parse_error;
            if (!sym_set(varname, value)) return E_BADEF;
        }
        break;
    case K_IF:
        push(exec);
        if (exec) {
            if ((value = parse_expression(0))) {
                exec = value->val.ival;
            }
            else {
                return parse_error;
            }
        }
        break;
    case K_ELSE:
        if (top()) exec = !exec;
        break;
    case K_ELIF:
        elifcnt++;
        if (top()) exec = !exec;
        push(exec);
        if (exec) {
            if ((value = parse_expression(0))) {
                exec = value->val.ival;
            }
            else {
                return parse_error;
            }
        }
        break;
    case K_END:
        do {
            exec = pull();
        } while (elifcnt--);
        elifcnt = 0;
        break;
    case K_INC:
        if (exec) {
            strncpy(arg, lex.tok, argsize);
        }
        else
            rval = A_NOP;
        break;
    case K_ON:
    case K_OFF:
        if (!exec) rval = A_NOP;
        break;
    case K_CTL:
        if (exec) {
            cchar = lex.tok[0];
        }
        else
            rval = A_NOP;
        break;
    case K_MSG:
    case K_ERR:
        if (!exec)
            rval = A_NOP;
        else {
            if (lex.sym != S_STR) return E_STREXP;
            strncpy(arg, lex.tok, argsize);
        }
        break;
    case K_DUMPSYM:
        break;
    }
    return rval?rval:(exec?A_COPY:A_SKIP);
}
/*-------------------------------------------------------------------------
 * return message string matching error code
 */

const char *
parse_geterrmsg(int e)
{
    int i;

    i = e - E_NUMBER;
    if (i > MAX_ENUM || i < 0) i = 0;
    return errtab[i];
}
