/*
 * $Id: strutil.c,v 1.2 2004/10/30 13:38:23 mark Exp $
 *
 * NAME
 *     strutil routines
 *
 * SYNOPSIS
 *     int strcont(char *s, char *w);
 *         Returns TRUE if all the letters in w can be found in s
 *     void strlwr(char *s);
 *         Converts all characters in a string to lower case (in place)
 *     void strrem(char *s,char *w,char *p);
 *         Removes all the characters in w from s, putting the remaining
 *         characters in p.
 *     void strsort(char *s,char *t);
 *         Sorts characters in string s into alphabetic order.  Sorted
 *         characters are returned via the t char *argument.
 *     void strsortf(char *s,char *t);
 *         Sorts characters in string s into reverse frequency order.
 *         String s must contain lower case letters only (a-z).
 *         Sorted characters are returned via the t char *argument.
 *
 * DESCRIPTION
 *     Contains a number of string-based routines to support
 *     anagram and related programs.
 *
 * NOTES
 *     The character frequencies used in srtsortf are from "The Code
 *     Book" by Simon Singh.
 *
 * MODIFICATION HISTORY
 * Mnemonic    Date    Rel Who
 * STRUTIL     01Oct06 1.0 MPW
 *     Written.
 *
 * Copyright (C) 2001,2004 Mark Willson.
 *
 *
 * This file is part of the maf program.
 *
 * The maf program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The maf program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <ctype.h>
#include "strutil.h"

#define TRUE 1
#define FALSE 0

// Size of internal string buffers
#define BUFSZ 80

//  strcont - returns TRUE if all the letters of w can be found in s

int strcont(char *s, char *w)
{
    int i,l;
    char *p, buf[BUFSZ+1];

    strncpy(buf,s,BUFSZ);
    buf[BUFSZ+1] = '\0';
    l = strlen(w);
    for (i=0 ; i<l ; i++) {
        if ((p=strchr(buf,w[i])) == NULL) {
            return(FALSE);
        }
        else {
            // remove found character to avoid using it again
            *p = ' ';
        }

    }
    return(TRUE);
}

// strlwr - converts all characters in a string to lower case (in place)

void strlwr(s)
char *s;
{
    do {
        *s = tolower(*s);
    } while (*(s++));
}

//  strrem - removes all the characters in w from s, putting the remaining
//  characters in p.

void strrem(char *s, char *w, char *p)
{
    int i,l;
    char *pins,
        *cptr,
        buf[BUFSZ+1];

    l = strlen(s);
    pins = p;
    strncpy(buf,w,BUFSZ);
    buf[BUFSZ+1] = '\0';

    for (i=0 ; i<l ; i++) {
        if ((cptr=strchr(buf,s[i])) == NULL) {
            *(pins++) = s[i];
        }
        else {
            // remove char from word to avoid matching it again
            *cptr = ' ';
        }
    }
    *pins = '\0';
}

//  strsort - sorts string t into alphabetic order and places
//  result in s

void strsort(char *t,char *s)
{
    int i,j,n,gap;
    char c;

    strcpy(s,t);
    n = strlen(t);
    for (gap=n/2; gap>0; gap/=2) {
        for (i=gap; i<n; i++) {
            for (j=i-gap; j>=0 && s[j]>s[j+gap]; j-=gap) {
                c = s[j];
                s[j] = s[j+gap];
                s[j+gap] = c;
            }
        }
    }
}

//  strsortf - sorts characters in string into reverse frequency order

void strsortf(char *t,char *s)
{
    // cfreq array holds character frequencies (per thou) in a-z order
    int cfreq[] = {
    //  a, b, c, d, e,  f, g, h, i, j, k, l, m,
        82,15,28,43,127,22,20,61,70, 2, 8,40,24,
    //  n, o, p, q, r, s, t, u, v, w, x, y, z
        67,75,19, 1,60,63,91,28,10,24, 2,20, 1
    };
    int i,j,n,gap;
    char c;

    strcpy(s,t);
    n = strlen(t);
    for (gap=n/2; gap>0; gap/=2) {
        for (i=gap; i<n; i++) {
            for (j=i-gap; j>=0 && cfreq[s[j]-'a']>cfreq[s[j+gap]-'a']; j-=gap) {
                c = s[j];
                s[j] = s[j+gap];
                s[j+gap] = c;
            }
        }
    }
}
